/*
 *  This file is part of "TweetyProject", a collection of Java libraries for
 *  logical aspects of artificial intelligence and knowledge representation.
 *
 *  TweetyProject is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 3 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *  Copyright 2016 The TweetyProject Team <http://tweetyproject.org/contact/>
 */
package org.tweetyproject.agents.sim;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.tweetyproject.agents.AbstractProtocol;
import org.tweetyproject.agents.Agent;
import org.tweetyproject.agents.MultiAgentSystem;
import org.tweetyproject.agents.ProtocolTerminatedException;

/**
 * This class implements a game simulator. It takes some agent and multi-agent system generators
 * and runs a series of simulations. In each simulation the winner of the game is determined and
 * recorded.
 * @author Matthias Thimm
 * @param <T> The actual type of agents.
 * @param <S> The actual type of protocols.
 * @param <R> The actual type of the multi-agent system.
 */
public class GameSimulator<S extends AbstractProtocol & GameProtocol, T extends Agent, R extends MultiAgentSystem<T>> {
	
	/** The multi-agent system generator. */
	private MultiAgentSystemGenerator<T,R> masGenerator;
	/** The protocol generator. */
	private ProtocolGenerator<S,T,R> protGenerator;
	/** The agent generators. */
	private List<AgentGenerator<T,R>> agentGenerators;
	
	/**
	 * Creates a new GameSimulator for the given
	 * MultiAgentSystemGenerator and AgentGenerators.
	 * @param masGenerator a MultiAgentSystemGenerator 
	 * @param protGenerator a protocol generator
	 * @param agentGenerators some AgentGenerators.
	 */
	public GameSimulator(MultiAgentSystemGenerator<T,R> masGenerator, ProtocolGenerator<S,T,R> protGenerator, List<AgentGenerator<T,R>> agentGenerators){
		this.masGenerator = masGenerator;
		this.agentGenerators = agentGenerators;
		this.protGenerator = protGenerator;
	}
	
	/**
	 * Simulates the game for the given number of
	 * repetitions and returns a map indicating
	 * which agent generator won how often.
	 * @param repetitions the number of repetitions.
	 * @return a map which assigns to each agent generator
	 *  the number of times an agent generated by it won the game.
	 * @throws ProtocolTerminatedException if a protocol is asked
	 *  to perform a step but has already terminated. 
	 */
	public SimulationResult<S,T,R> run(int repetitions) throws ProtocolTerminatedException{
		SimulationResult<S,T,R> result = new SimulationResult<S,T,R>(this.agentGenerators);
		for(int i = 0; i < repetitions; i++){
			Map<Agent,AgentGenerator<T,R>> a2ag = new HashMap<Agent,AgentGenerator<T,R>>();
			SimulationParameters params = new SimulationParameters();
			R mas = this.masGenerator.generate(params);
			// create agents
			for(AgentGenerator<T,R> ag: this.agentGenerators){
				T a = ag.generate(mas,params);
				mas.add(a);
				a2ag.put(a, ag);
			}		
			S prot = this.protGenerator.generate(mas,params);
			mas.execute(prot);	
			if(prot.hasWinner()){
				Agent winner = prot.getWinner();
				Map<AgentGenerator<T,R>,Double> utilities = new HashMap<AgentGenerator<T,R>,Double>();
				for(Agent a: a2ag.keySet())
					utilities.put(a2ag.get(a), prot.getUtility(a));
				result.addEntry(a2ag.get(winner), utilities);
			}
		}		
		return result;
	}
}
