/*
 *  This file is part of "TweetyProject", a collection of Java libraries for
 *  logical aspects of artificial intelligence and knowledge representation.
 *
 *  TweetyProject is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version 3 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *  Copyright 2020 The TweetyProject Team <http://tweetyproject.org/contact/>
 */
package org.tweetyproject.arg.dung.reasoner;

import java.util.Collection;
import java.util.HashSet;

import org.tweetyproject.arg.dung.syntax.Argument;
import org.tweetyproject.arg.dung.syntax.DungTheory;
import org.tweetyproject.commons.InferenceMode;

/**
 * Implements the EEE approach ("exhaustively enumerate extensions") to
 * determine the set of acceptable arguments of an AF, cf. [Thimm, Cerutti, Vallati; COMMA 2020].
 * It uses another extension-based reasoner to retrieve all extensions
 * and takes the union/intersection.
 * 
 * @author Matthias Thimm
 *
 */
public class EeeAcceptabilityReasoner extends AbstractAcceptabilityReasoner {

	private AbstractExtensionReasoner reasoner;
	private InferenceMode inferenceMode;
	
	/**
	 * Creates a new EeeAcceptabilityReasoner.
	 * @param reasoner the internal reasoner used.
	 * @param inferenceMode the inference mode.
	 */
	public EeeAcceptabilityReasoner(AbstractExtensionReasoner reasoner, InferenceMode inferenceMode) {
		this.reasoner = reasoner;
		this.inferenceMode = inferenceMode;
	}
	
	@Override
	public Collection<Argument> getAcceptableArguments(DungTheory aaf) {
		Collection<Argument> result = new HashSet<Argument>();
		if(this.inferenceMode.equals(InferenceMode.CREDULOUS))
			for(Collection<Argument> extension: this.reasoner.getModels(aaf))
				result.addAll(extension);
		else {
			result.addAll(aaf);
			for(Collection<Argument> extension: this.reasoner.getModels(aaf))
				result.retainAll(extension);			
		}			
		return result;
	}	
    /**
     * this method always returns true because the solver is native
     */
	@Override
	public boolean isInstalled() {
		return true;
	}
}
