/*
 * Copyright 1997-2013 Optimatika (www.optimatika.se)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.ojalgo.matrix.store;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.ojalgo.ProgrammingError;
import org.ojalgo.access.Access1D;
import org.ojalgo.scalar.Scalar;

/**
 * A merger of two {@linkplain MatrixStore} instances by placing one store below the other. The two matrices must have
 * the same number of columns. The columns of the two matrices are logically merged to form new longer columns.
 * 
 * @author apete
 */
public final class AboveBelowStore<N extends Number> extends DelegatingStore<N> {

    private final MatrixStore<N> myLowerStore;
    private final int myRowSplit;

    public AboveBelowStore(final MatrixStore<N> base, final MatrixStore<N> aLowerStore) {

        super((int) (base.countRows() + aLowerStore.countRows()), (int) base.countColumns(), base);

        myLowerStore = aLowerStore;
        myRowSplit = (int) base.countRows();
    }

    @SuppressWarnings("unused")
    private AboveBelowStore(final MatrixStore<N> aBase) {

        this(aBase, null);

        ProgrammingError.throwForIllegalInvocation();
    }

    /**
     * @see org.ojalgo.matrix.store.MatrixStore#doubleValue(long, long)
     */
    public double doubleValue(final long row, final long column) {
        return (row >= myRowSplit) ? myLowerStore.doubleValue(row - myRowSplit, column) : this.getBase().doubleValue(row, column);
    }

    public N get(final long row, final long column) {
        return (row >= myRowSplit) ? myLowerStore.get(row - myRowSplit, column) : this.getBase().get(row, column);
    }

    public boolean isLowerLeftShaded() {
        return false;
    }

    public boolean isUpperRightShaded() {
        return this.getBase().isUpperRightShaded();
    }

    @Override
    public MatrixStore<N> multiplyRight(final Access1D<N> rightMtrx) {

        final Future<MatrixStore<N>> tmpBaseFuture = this.executeMultiplyRightOnBase(rightMtrx);

        final MatrixStore<N> tmpLower = myLowerStore.multiplyRight(rightMtrx);

        try {
            return new AboveBelowStore<N>(tmpBaseFuture.get(), tmpLower);
        } catch (final InterruptedException | ExecutionException ex) {
            return null;
        }
    }

    public Scalar<N> toScalar(final int row, final int column) {
        return (row >= myRowSplit) ? myLowerStore.toScalar(row - myRowSplit, column) : this.getBase().toScalar(row, column);
    }

}
